--------------------------------------------------------------------------------
-- DIO2 board LCD driver
-- LCD startup sequence driver
--
-- Michal TRS
-- trsm1@fel.cvut.cz 
--------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;


entity lcd_init_drv is
  Port ( 
   CLK      : in std_logic;
   AS_RESET : in std_logic;
   -- ---
   INIT     : in  std_logic;
   RDY      : out std_logic;

   DATA     : out std_logic_vector(7 downto 0);
   RW       : out std_logic;
   RS       : out std_logic;
   DATA_VLD : out std_logic;
   WR_RDY   : in  std_logic);
end lcd_init_drv;

architecture lcd_init_drv_body of lcd_init_drv is

   type t_init_states is (st_wait, st_func, st_w1, st_ctrl, st_w2, st_clear, st_w3, st_mode, st_rdy);

   signal cur_state   : t_init_states;
   signal next_state  : t_init_states;

   -- 20 ns = 50 MHz , max delay 1.52 us => 17 bit
   signal cnt        : std_logic_vector(16 downto 0);
   signal cnt_ce     : std_logic;
   signal cnt_load   : std_logic;
   signal cnt_data   : std_logic_vector(16 downto 0);
   signal cnt_zero   : std_logic; 

begin


   -- ---------------------------------
   -- Driving FSM
   -- ---------------------------------

   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cur_state <= st_wait;
      elsif CLK = '1' and CLK'event then
         cur_state <= next_state;
      end if;
   end process;

   -- next state logic
   process(cur_state, INIT, WR_RDY, cnt_zero)
   begin
      next_state <= cur_state;

      case cur_state is
         when st_wait =>
            if INIT = '1' then
               next_state <= st_func;
            end if;
         when st_func =>
            if WR_RDY = '1' then
               next_state <= st_w1;
            end if;            
         when st_w1 =>
            if cnt_zero = '1' then
              next_state <= st_ctrl;
            end if;
         when st_ctrl =>
            if WR_RDY = '1' then
               next_state <= st_w2;
            end if;            
         when st_w2 =>
            if cnt_zero = '1' then
              next_state <= st_clear;
            end if;
         when st_clear =>
            if WR_RDY = '1' then
               next_state <= st_w3;
            end if;                        
         when st_w3 =>
            if cnt_zero = '1' then
              next_state <= st_mode;
            end if;
         when st_mode =>
            if WR_RDY = '1' then
               next_state <= st_rdy;
            end if;              
         when st_rdy =>
            next_state <= st_rdy;
      end case;
   end process;

   -- output logic
   process(cur_state, INIT, WR_RDY, cnt_zero)
   begin
      RDY      <= '0';
      cnt_data <= (others => '0');
      cnt_ce   <= '0';
      cnt_load <= '0';

      RS       <= '0';
      RW       <= '0';
      DATA     <= (others => '0');
      DATA_VLD <= '0';

      case cur_state is
         when st_wait =>
            if INIT = '1' then
               -- Function set
               RS       <= '0';
               RW       <= '0';
               DATA     <= "00111000";
               DATA_VLD <= '1';
            end if;

         when st_func =>
            if WR_RDY = '1' then
               cnt_data <= conv_std_logic_vector(1850,17); -- 37us
               cnt_load <= '1';
            end if;  
                      
         when st_w1 =>
            cnt_ce <= '1';
            if cnt_zero = '1' then
               -- Display control set
               RS       <= '0';
               RW       <= '0';
               DATA     <= "00001111";
               DATA_VLD <= '1';
            end if;

         when st_ctrl =>
            if WR_RDY = '1' then
               cnt_data <= conv_std_logic_vector(1850,17); -- 37us
               cnt_load <= '1';
            end if;  

         when st_w2 =>
            cnt_ce <= '1';
            if cnt_zero = '1' then
               -- Display clear
               RS       <= '0';
               RW       <= '0';
               DATA     <= "00000001";
               DATA_VLD <= '1';
            end if;

         when st_clear =>
            if WR_RDY = '1' then
               cnt_data <= conv_std_logic_vector(76000,17); -- 1.52ms
               cnt_load <= '1';
            end if;

         when st_w3 =>
            cnt_ce <= '1';
            if cnt_zero = '1' then
               -- Entry mode set
               RS       <= '0';
               RW       <= '0';
               DATA     <= "00000111";
               DATA_VLD <= '1';
            end if;

         when st_mode =>
            if WR_RDY = '1' then
               cnt_data <= conv_std_logic_vector(1850,17); -- 37us
               cnt_load <= '1';
            end if;
             
         when st_rdy =>
            if cnt_zero = '1' then
               RDY <= '1';
            else 
               cnt_ce <= '1';
            end if;
      end case;
   end process;



   -- -------------------------------- 
   -- counter
   -- --------------------------------

   icnt:process(CLK,cnt_ce,cnt_load)
   begin
      if CLK = '1' and CLK'event then
         if cnt_load = '1' then
            cnt <= cnt_data; 
         elsif cnt_ce = '1' then
            cnt <= cnt - 1;  
         end if;    
      end if;
   end process;
   
   cnt_zero <= '1' when cnt = conv_std_logic_vector(0,17)
          else '0';


end lcd_init_drv_body;
