/*
  hci.c
*/
/* 
   BlueMP3 firmware (c) 2004 by Till Harbaum, harbaum@beecon.de
*/


#include <compat/deprecated.h>

#include "types.h"
#include "debug.h"

#include "uart.h"
#include "hci.h"
#include "hci_uart.h"
#include "hci_event.h"
#include "hci_info.h"
#include "hci_acl.h"
#include "hci_con.h"
#include "utils.h"
#include "cpu.h"

#include <stdarg.h>

u08_t hci_device_name[] = DEVICE_NAME;

static u08_t hci_decode(u16_t cmd, void *data) {
  DEBUG_HCI("hci_decode() start\n");
  u08_t result = HCI_DECODE_CONT;  /* command has not been ack'd yet */
  /* fetch hci message type byte */
  switch(hci_uart_get()) {
    case HCI_ACL:
      hci_acl_decode(cmd);
      break;

    case HCI_EVT:
      result = hci_event(cmd, data);
      break;

    default:
      DEBUG_HCI("Received invalid HCI UART msg type!\n");    
      //do {
	  while (uart_input_avail()) uart_getc();
	    //cpu_delay(CPU_MILLISEC(50));
      //} while (uart_input_avail());
	  DEBUG_HCI("all uart data dropped!\n");
	  break;
  }
  DEBUG_HCI("hci_decode() end\n");
  return result;
}


/* start the transmission of a hci command */
static void hci_cmd_start(void) {
  /* wait for free command buffers */  
  while(!hci_further_commands)
    hci_process(HCI_PROCESS_WAIT_CMD);

  hci_uart_put(HCI_CMD);   // this will be a hci command
}


#define GET_U16LE(a)  ((a)[0]+256u*(a)[1])



/* wait for the command to be processed */
static bool_t hci_wait(u08_t *cmd, void *data) {
  u08_t ret;
  u16_t timeout = 50000;

  /* wait for serial data (with timeout) */
  while((!uart_input_avail())&&(timeout))
    timeout--;

  if(!timeout) {
    DEBUG_HCI("command reply timeout!\n");

    return FALSE;       // retry due to timeout
  }

  /* wait for command complete, this command blocks and if no */
  /* answer is received within the watch dog timeout, the device */
  /* is being reset. this is ok, since _all_ commands should */
  /* trigger a reply within few milliseconds. see hci_acl.c for */
  /* the only exception to this rule. */
  while((ret = hci_decode(GET_U16LE(cmd), data)) == HCI_DECODE_CONT);

  /* return FALSE on error (hardware out of sync) */
  return(ret == HCI_DECODE_OK);
}


/* send a byte sequence */
void hci_send_seq(u08_t *data, u08_t len) {
  if(data) hci_uart_send(data, len); 
  else     hci_uart_send_zero(len);
}


/* send a hci command consisting of one single byte sequence */
void hci_send_cmd(u08_t *cmd, u08_t len, u08_t par_num, ...) {
  DEBUG_HCI("hci_send_cmd() start\n");
  
  va_list ap;
  u08_t *pptr, cnt;
  u16_t plen;
  void  *data = NULL;

  do {
    va_start(ap, par_num);

    /* send cmd */
    hci_cmd_start();
    hci_uart_send(cmd, len);

    /* is a result expected? */
    if(par_num & HCI_PARMS) {
      /* yes, get buffer address */
      data = va_arg(ap, void *);
      par_num &= ~HCI_PARMS;
    }

    /* send parameters */
    for(cnt=0;cnt<par_num;cnt++) {

      /* get address and length */
      pptr = va_arg(ap, u08_t *);
      plen = va_arg(ap, int);

      /* and send it from flash, eeprom, ram, ... */
      switch(plen & HCI_FLAG_MASK) {
		case HCI_RAM:
		  hci_uart_send(pptr, plen);  // no mask req., since HCI_RAM == 0
		  break;
		
		case HCI_ZERO:
		  hci_uart_send_zero(plen & HCI_LEN_MASK);
		  break;
      }
    }

    va_end(ap);

    /* wait for cmd to be acked, hci_wait returns FALSE on hardware */ 
    /* error. This indicates, that the hci layer is out of sync and */
    /* the last command is very likely lost */
  } while(!hci_wait(cmd, data));
  DEBUG_HCI("hci_send_cmd() end\n");
}


/* set name from eeprom */
void hci_set_name(void) {
  static u08_t hci_cmd_chname[] = { 
    OPCODE(OGF_HOST_CTL, OCF_CHANGE_LOCAL_NAME), 248 };

  hci_send_cmd(hci_cmd_chname, sizeof(hci_cmd_chname), 2,
	       hci_device_name, sizeof(hci_device_name),
	       NULL,            HCI_ZERO | (248 - sizeof(hci_device_name)));
}


bool_t hci_reset(void) {
  u08_t to = 100;  // 100*10ms timeout
  static u08_t hci_cmd_reset[] = { OPCODE(OGF_HOST_CTL, OCF_RESET), 0 };
  
  DEBUG_HCI("hci_reset...\n");
  /* send reset cmd */
  hci_cmd_start();
  hci_uart_send(hci_cmd_reset, sizeof(hci_cmd_reset));

  /* and wait for some answer */
  while(!uart_input_avail()) {
	cpu_delay(CPU_MILLISEC(10));
    if(!to--) return FALSE;
  }
  
  /* there's some reply */
  return hci_wait(hci_cmd_reset, NULL);
}


/* send all commands necessary to set up the bluetooth interface */
void hci_init(void) {
  static u08_t hci_cmd_write_cod[] = { 
    OPCODE(OGF_HOST_CTL, OCF_WRITE_COD), 3, MK_SEQ3(DEVICE_CLASS) };

  static u08_t hci_cmd_inq_enable[] = {
    OPCODE(OGF_HOST_CTL, OCF_WRITE_SCAN_ENABLE), 1,
    PAGE_SCAN_ENABLE | INQUIRY_SCAN_ENABLE };

  /* setup/clear connection structures etc */
  hci_con_init();

  /* send reset command */
  if(!hci_reset()) {
    DEBUG_HCI("reset timed out! Exit\n");
	return;
  }

  DEBUG_HCI("bluetooth reset successful\n");
  
  
  /* set name of device */
  DEBUG_HCI("setting name\n");
  hci_set_name();

  /* set class of device */
  DEBUG_HCI("setting COD\n");
  hci_send_cmd(hci_cmd_write_cod, sizeof(hci_cmd_write_cod), 0);

  /* enable the device to be discovered */
  DEBUG_HCI("write page enable\n");
  hci_send_cmd(hci_cmd_inq_enable, sizeof(hci_cmd_inq_enable), 0);

  /* send initial init commands */
  hci_info_init();
}


void hci_process(u16_t cmd) {
  /* process data if byte on uart waiting or buffers in use */
  if(uart_input_avail()) {
    /* decode without a pending command */
    hci_decode(cmd, NULL);
  }
}
