/*
  hci_acl.c

  ACL layer of MicroBlue
*/
/* 
   BlueMP3 firmware (c) 2004 by Till Harbaum, harbaum@beecon.de
*/


#include "types.h"
#include "hci.h"
#include "hci_acl.h"
#include "hci_uart.h"
#include "hci_con.h"
#include "l2cap.h"
#include "debug.h"


u16_t acl_expect, acl2send;  /* acl state holding */

u08_t hci_acl_get(void) {
  u08_t byte;
  
  /* if no acl data directly available, wait for next packet */
  while(!acl_expect)
    hci_process(HCI_PROCESS_IDLE);

  acl_expect--;              // one byte will be returned
  byte = hci_uart_get();

  return byte;
}

/* get a 16 bit value from acl payload */
u16_t hci_acl_get_u16(void) {
  return hci_acl_get() + 256 * hci_acl_get();
}

/* drop num bytes acl payload */
void hci_acl_drop(u08_t num) {
  while(num--) hci_acl_get();
}


/* number of ACL packets that might still */ 
/* be sent and maximum size of acl packets */
static u16_t acl_num;
static u16_t acl_size; 

u16_t hci_acl_fsize(void) {
  return acl_size;
}

/* set initial buffer values derived from bt module */
void hci_acl_set(u16_t num, u16_t size) {
  acl_size = size;
  acl_num  = num;
}

/* bt module has indicated, that buffers have been freed again */
void hci_acl_pkt_completed(u16_t num) {
  acl_num += num;
}

/* decode an incoming acl message */
void hci_acl_decode(u16_t cmd) {
  u16_t handle = hci_uart_get_u16();
  u16_t length = hci_uart_get_u16();
  hci_acl_con_t *acl;

  /* get matching acl entry */
  acl = hci_get_acl_slot(handle & HCI_ACL_HND_MASK);

  if(!length) {
    DEBUG_HCI("ignoring empty ACL packet\n");
    return;
  }

  /* we are waiting for buffer, drop payload */
  if((cmd == HCI_PROCESS_WAIT_ACL)||(cmd == HCI_PROCESS_WAIT_CMD) ) {
    hci_uart_drop(length);    
    return;
  }

  /* unknown acl handle, ignore payload */
  if(!acl) {
    hci_uart_drop(length);    
    return;
  }

  /* forward data to correct layer */
  switch(handle & HCI_ACL_CMD_MASK) {
    case L2CAP_START:
      l2cap_decode(acl, length);
      break;

    case L2CAP_CONTINUE:
      /* continuation, but no l2cap transfer in progress? drop data */
      if(!acl->l2cap_expect) hci_uart_drop(length);
      else    	             acl_expect = length;

      break;

    default:
      /* unknown packet type */
      hci_uart_drop(length);
  }
}

/* send hci acl header */
void hci_acl_send(hci_acl_con_t *acl, u16_t len, bool_t l2start) {

  /* wait for free acl slots if necessary */
  while(!acl_num) hci_process(HCI_PROCESS_WAIT_ACL);

  acl_num--;
  hci_uart_put(HCI_ACL);

  if(l2start) hci_uart_put_u16(acl->handle | L2CAP_START);
  else        hci_uart_put_u16(acl->handle | L2CAP_CONTINUE);

  hci_uart_put_u16(len);
}
