/*
  hci_con.c

  hci connection handling 
*/
/* 
   BlueMP3 firmware (c) 2004 by Till Harbaum, harbaum@beecon.de
*/


#include "types.h"
#include "debug.h"

#include "hci.h"
#include "hci_con.h"
#include "hci_callback.h"
#include "hci_uart.h"
#include "l2cap.h"
#include "utils.h"

/* global info about the existing acl connection(s) */
hci_acl_con_t acl_con[HCI_ACL_CLIENTS];

#define HANDLE_FREE      0xffff   // marker for unused handle
#define HANDLE_PREALLOC  0xfffe   // marker for pre-allocated handle

/* init connection structure */
void hci_con_init(void) {
  u08_t i;

  /* mark handles as unused */
  for(i=0;i<HCI_ACL_CLIENTS;i++)
    acl_con[i].handle = HANDLE_FREE;
}

/* get a free acl entry, return NULL if no available */
static hci_acl_con_t *hci_get_free_acl_slot(void) {
  u08_t i;

  /* search for free handle */
  for(i=0;i<HCI_ACL_CLIENTS;i++) {

    if(acl_con[i].handle == HANDLE_FREE) {
      DEBUG_HCI("free acl slot is available\n");

      /* clear the newly allocated slot (contains a ppp buffer, */
      /* therefore is > 256) */
      utils_memzero(&acl_con[i], sizeof(hci_acl_con_t));
      acl_con[i].handle = HANDLE_FREE;   // not allocated yet

      return &acl_con[i];
    }
 }

  DEBUG_HCI("no free acl slot available!\n");
  return NULL;
}

/* get the acl entry associated to a given acl handle */
hci_acl_con_t *hci_get_acl_slot(u16_t handle) {
  u08_t i;

  /* search for the right handle */
  for(i=0;i<HCI_ACL_CLIENTS;i++)
    if(acl_con[i].handle == handle) 
      return &acl_con[i];

  return NULL;
}

void hci_event_connection_request(void) {
  static u08_t hci_cmd_accept[] = { 
    OPCODE(OGF_LINK_CTL, OCF_ACCEPT_CONN_REQ), 7 };

  static u08_t hci_cmd_reject[] = { 
    OPCODE(OGF_LINK_CTL, OCF_REJECT_CONN_REQ), 7 };

  hci_con_req_t con_req;
  u08_t code;
  bool_t addr_ok;

  hci_uart_get_n((u08_t*)&con_req, sizeof(con_req));

  DEBUG_HCI("connection request\n");

  /* check for 00:00:00:00:00:00 hw address and reject it */
  /* observed several times with a palm tungsten, reboot of */
  /* palm fixes the problem */
  addr_ok = 
    con_req.bd_addr[0] || con_req.bd_addr[1] || 
    con_req.bd_addr[2] || con_req.bd_addr[3] || 
    con_req.bd_addr[4] || con_req.bd_addr[5];

  if((hci_uart_get() != LINK_TYPE_ACL)||
     (!hci_get_free_acl_slot()) || (!addr_ok)) {
    DEBUG_HCI("no ACL, free handles or illegal bdaddr, rejecting\n");
    
    if(!addr_ok) code = HCI_ERROR_REJECT_SECURITY;
    else         code = HCI_ERROR_REJECT_LIMIT;

    /* reject non-acl connection or while con already exists */
    hci_send_cmd(hci_cmd_reject,   sizeof(hci_cmd_reject), 2,
		 &con_req.bd_addr, 6,
		 &code,            1);
  } else {
    DEBUG_HCI("accepting\n");
    code = HCI_ROLE_REMAIN_SLAVE;

    hci_send_cmd(hci_cmd_accept,   sizeof(hci_cmd_accept), 2,
		 &con_req.bd_addr, 6,
		 &code,            1);
  }
}

void hci_event_connection_complete(void) {
  hci_con_comp_t con_comp;
  hci_acl_con_t *acl = NULL;

  hci_uart_get_n((u08_t*)&con_comp, sizeof(con_comp));

  DEBUG_HCI("connection complete\n");

  /* setup acl structure if everything is fine */
  if(!con_comp.errcode) {

    /* not entry from own connection request found, allocating new one */
    if(!acl)
      acl = hci_get_free_acl_slot();

    if(acl) {
      acl->handle = con_comp.handle;
      utils_memcpy(acl->bdaddr, con_comp.bd_addr, 6);

      /* clear all l2cap connection entries of this acl connection */
      utils_memzero((u08_t*)acl->l2con, MAX_L2CON * sizeof(hci_l2con_t));

      /* contact application */
      hci_internal_callback(HCI_EVT_ACL_CONNECTED, acl);
    } else {
      con_comp.errcode = HCI_ERROR_MB_NO_SLOT;
      hci_internal_callback(HCI_EVT_ACL_CONNECTION_FAILED, acl);
    }
  } else 
    hci_internal_callback(HCI_EVT_ACL_CONNECTION_FAILED, acl);
}

void hci_event_disconnection_complete(void) {
  hci_disc_comp_t disc_comp;
  hci_acl_con_t *acl;
  
  hci_uart_get_n((u08_t*)&disc_comp, sizeof(disc_comp));

  DEBUG_HCI("disconnection complete\n");

  acl = hci_get_acl_slot(disc_comp.handle);
  if(acl) {
    /* check if there's still an existing l2cap connection */
    /* since the link may have been lost by error */
    l2cap_link_broken(acl);

    /* contact application */
    hci_internal_callback(HCI_EVT_ACL_DISCONNECTED, acl);

    /* whatever the reason was, consider the connection gone */
    acl->handle = HANDLE_FREE;
  }
}

/* search for a free l2cap slot */
u08_t hci_con_new_l2cap_entry(hci_acl_con_t *acl) {
  u08_t slot;

  for(slot=0;slot<MAX_L2CON;slot++)
    if(!acl->l2con[slot].psm) 
      return slot;

  return HCI_NO_SLOT;
}
