/*
  hci_event.c

  hci layer event processing
*/
/* 
   BlueMP3 firmware (c) 2004 by Till Harbaum, harbaum@beecon.de
*/



#include "types.h"
#include "debug.h"

#include "hci.h"
#include "hci_event.h"
#include "hci_uart.h"
#include "hci_con.h"
#include "hci_info.h"
#include "hci_acl.h"
#include "utils.h"

u08_t hci_further_commands = 1;


static u08_t hci_event_command_complete(u16_t cmd, u08_t len, void *data) {
  u08_t ret = HCI_DECODE_CONT;

  hci_further_commands = hci_uart_get();

  /* verify the two command bytes */
  if(cmd == hci_uart_get_u16())
    ret = HCI_DECODE_OK;

  /* process additional data */

  /* we we weren't waiting for a command, just return ok */
  if(cmd <= HCI_PROCESS_WAIT_CMD)
    ret = HCI_DECODE_OK;

  /* still bytes to read */
  if(len>3) {

    /* is the data pointer valid and the command the expect one? */
    if(data && (ret == HCI_DECODE_OK)) {
      hci_uart_get_n(data, len-3);
    } else {
      hci_uart_drop(len-3);
    }
  }

  return ret;
}


static u08_t hci_event_command_status(u16_t cmd, u08_t len) {
  u08_t ret = HCI_DECODE_CONT;
  hci_uart_get();

  hci_further_commands = hci_uart_get();

  /* verify the two command bytes */
  if(cmd == hci_uart_get_u16())
    ret = HCI_DECODE_OK;

  return ret;
}


void hci_event_number_of_complete_packets(void) {
  u08_t i, handles = hci_uart_get();

  for(i=0;i<handles;i++) {
    hci_uart_drop(2);                // don't care about the handles
    hci_acl_pkt_completed(hci_uart_get_u16());
  }
}


void hci_event_pin_request(void)
{
  static u08_t pin[] = {'0','0','0','0'};
  u08_t pinLength = sizeof(pin);
  u08_t bd_addr[6];
  static u08_t hci_cmd_pin[] = {
    OPCODE(OGF_LINK_CTL, OCF_PIN_CODE_REQUEST_REPLY), 23};
  
  hci_uart_get_n(bd_addr, sizeof(bd_addr));

  hci_send_cmd(hci_cmd_pin, sizeof(hci_cmd_pin), 4,
		 bd_addr, sizeof(bd_addr),
		 &pinLength, 1, 
		 pin, sizeof(pin),
		 NULL, HCI_ZERO | (16 - sizeof(pin)));
}


void hci_event_link_key_request(void)
{
  static u08_t hci_cmd_pin[] = {
    OPCODE(OGF_LINK_CTL, OCF_LINK_KEY_NEGATIVE_REPLY), 6};
  u08_t bd_addr[6];
  
  hci_uart_get_n(bd_addr, sizeof(bd_addr));

  hci_send_cmd(hci_cmd_pin, sizeof(hci_cmd_pin), 1, bd_addr, sizeof(bd_addr));
}



u08_t hci_event(u16_t cmd, void *data) {
  u08_t evt = hci_uart_get();
  u08_t len = hci_uart_get();

  switch(evt) {
    case EVT_COMMAND_COMPLETE:
      return hci_event_command_complete(cmd, len, data);
      break;

    case EVT_COMMAND_STATUS:
      return hci_event_command_status(cmd, len);
      break;

    case EVT_CONNECTION_REQUEST:
      hci_event_connection_request();
      break;

    case EVT_CONNECTION_COMPLETE:
      hci_event_connection_complete();
      break;

    case EVT_DISCONNECT_COMPLETE:
      hci_event_disconnection_complete();
      break;


    case EVT_NUMBER_OF_COMPLETE_PKTS:
      hci_event_number_of_complete_packets();
      break;

    case EVT_HARDWARE_ERROR:
      DEBUG_HCI("hardware error (hci out of sync?)\n");
      hci_uart_drop(len);
      return HCI_DECODE_ERR;
      break;

    case EVT_PIN_CODE_REQUEST:
	  if (len == 6) {
	    DEBUG_HCI("PIN request, reply \"0000\"\n");
		hci_event_pin_request();
	  } else hci_uart_drop(len);
	  break;

	case EVT_LINK_KEY_REQUEST:
	  if (len == 6) {
	    DEBUG_HCI("link key request (key wasn't stored)\n");
		hci_event_link_key_request();
	  } else hci_uart_drop(len);
	  break;
	  

    default:
      /* just ignore unsupported hci events in this simple BlueMP3 demo code */
      hci_uart_drop(len);
      break;
  }
  return HCI_DECODE_CONT;
}
