#include <dev/board.h>

#include <sys/heap.h>
#include <sys/thread.h>
#include <sys/timer.h>
#include <sys/socket.h>
#include <sys/confnet.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>

#include <arpa/inet.h>

#include <pro/dhcp.h>

#include "BT/hci.h"
#include "BT/hci_callback.h"
#include "BT/hci_con.h"
#include "BT/l2cap.h"

#define MY_MAC      "\x00\x06\x98\x30\x00\x35"
#define MY_IPADDR   "192.168.1.100"
#define MY_IPMASK   "255.255.255.0"
#define TCPPORT     23
#define BUFFERSIZE  128


FILE *efp = NULL;    // ethernet file pointer
FILE *ufp = NULL;    // uart file pointer
hci_l2con_t * l2con; // Bluetooth l2cap pointer

int eth_connected = 0;
int bt_connected = 0;

/*
 * Transfer data from input stream to output stream.
 */
void eth2bt(void)
{
	int cnt;
    char *buff;

    buff = malloc(BUFFERSIZE);
    while (eth_connected) {
        if ((cnt = fread(buff, 1, BUFFERSIZE, efp)) <= 0) {
			eth_connected = 0;
            break;
		}
		if (bt_connected) {
			l2cap_send(l2con, buff, cnt);
			fflush(ufp);
		}
    }
    free(buff);
}

/*
 * Transfer data from input stream to output stream.
 */
void bt2eth(void)
{
    while (eth_connected) {
		hci_process(HCI_PROCESS_IDLE);
    }
}


/*
 * From Bluetooth to socket.
 */
THREAD(Receiver, arg)
{
    for (;;) {
        if (eth_connected) {
            NutThreadSetPriority(64);
			bt2eth();
			NutThreadSetPriority(128);
        }
        NutThreadYield();
    }
}


/*
 * Main application routine. 
 *
 * Nut/OS automatically calls this entry after initialization.
 */
int main(void)
{
    TCPSOCKET *sock;
    u_long baud = 115200;
	u_long timeout = 0;

    /*
     * Register our devices.
     */
    NutRegisterDevice(&DEV_UART, 0, 0);
    NutRegisterDevice(&DEV_ETHER, 0x8300, 5);

    /*
     * Setup the uart device.
     */
    ufp = fopen(DEV_UART_NAME, "r+b");
    _ioctl(_fileno(ufp), UART_SETSPEED, &baud);
	_ioctl(_fileno(ufp), UART_SETREADTIMEOUT, &timeout);

	/*
	 * Bluetooth init
	 */
	hci_init();

    /*
     * Setup the ethernet device. Use fixed IP. 
     */
	u_char my_mac[] = MY_MAC;
	u_long ip_addr = inet_addr(MY_IPADDR);
	u_long ip_mask = inet_addr(MY_IPMASK);
	NutNetIfConfig("eth0", my_mac, ip_addr, ip_mask);

    /*
     * Start a RS232 receiver thread.
     */
    NutThreadCreate("rcv", Receiver, NULL, 4096);

    /*
     * Now loop endless for connections.
     */
    eth_connected = 0;

    for (;;) {
        /*
         * Create a socket and listen for a client.
         */
        sock = NutTcpCreateSocket();
        NutTcpAccept(sock, TCPPORT);

        /*
         * Open a stdio stream assigned to the connected socket.
         */
        efp = _fdopen((int) sock, "r+b");
		eth_connected = 1;


		eth2bt();

        /*
         * Close the stream.
         */
        fclose(efp);
		efp = NULL;

        /*
         * Close our socket.
         */
        NutTcpCloseSocket(sock);
	}
}
