using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace FTPParser
{
    class Program
    {
        static private CSVReader csvr;
        static private StreamWriter sw;

        static private Dictionary<string, string> commands = new Dictionary<string,string>();
        static private Dictionary<string, string> replay_codes = new Dictionary<string,string>();


        static void init()
        {
            /* http://www.w3.org/Protocols/rfc959/4_FileTransfer.html *
             * http://www.networksorcery.com/enp/protocol/ftp.htm     */

            commands.Add("ABOR","Abort"); 
            commands.Add("ACCT","Account"); 
            commands.Add("ADAT","Authentication/Security Data");
            commands.Add("ALLO","Allocate"); 
            commands.Add("APPE","Append"); 
            commands.Add("AUTH","Authentication/Security Mechanism");
            commands.Add("CCC","Clear Command Channel"); 
            commands.Add("CDUP","Change to parent directory"); 
            commands.Add("CONF","Confidentiality Protected Command");
            commands.Add("CWD","Change working directory"); 
            commands.Add("DELE","Delete"); 
            commands.Add("ENC","Privacy Protected Command");
            commands.Add("EPRT","Extended Data port"); 
            commands.Add("EPSV","Extended Passive"); 
            commands.Add("FEAT","Feature");
            commands.Add("HELP","Help"); 
            commands.Add("LANG","Language negotiation"); 
            commands.Add("LIST","List");
            commands.Add("LPRT","Long data port"); 
            commands.Add("LPSV","Long passive"); 
            commands.Add("MDTM","File modification time");
            commands.Add("MIC","Integrity Protected Command"); 
            commands.Add("MKD","Make directory"); 
            commands.Add("MLSD","described in RFC 3659");
            commands.Add("MODE","Transfer mode"); 
            commands.Add("NLST","Name list"); 
            commands.Add("NOOP","No operation");
            commands.Add("OPTS","Options"); 
            commands.Add("PASS","Password"); 
            commands.Add("PASV","Passive mode");
            commands.Add("PBSZ","Protection Buffer Size"); 
            commands.Add("PORT","Data port"); 
            commands.Add("PROT","Data Channel Protection Level");
            commands.Add("PWD","Print working directory"); 
            commands.Add("QUIT","Logout"); 
            commands.Add("REIN","Reinitialize");
            commands.Add("REST","Restart of interrupted transfer"); 
            commands.Add("RETR","Retrieve"); 
            commands.Add("RMD","Remove directory");
            commands.Add("RNFR","Rename from"); 
            commands.Add("RNTO","Rename to"); 
            commands.Add("SITE","Site parameters");
            commands.Add("SIZE","File size"); 
            commands.Add("SMNT","Structure mount"); 
            commands.Add("STAT","Status");
            commands.Add("STOR","Store"); 
            commands.Add("STOU","Store unique"); 
            commands.Add("STRU","File structure");
            commands.Add("SYST","System"); 
            commands.Add("TYPE","Representation type"); 
            commands.Add("USER","User name"); 
            commands.Add("XSEM","Send, Mail if cannot"); 
            commands.Add("XSEN","Send to terminal");
        
            replay_codes.Add("110","Restart marker reply");
            replay_codes.Add("120","Service ready in nnn minutes.");
            replay_codes.Add("125","Data connection already open; transfer starting.");
            replay_codes.Add("150","File status okay; about to open data connection.");
            replay_codes.Add("200","Command okay.");
            replay_codes.Add("202","Command not implemented");
            replay_codes.Add("211","System status, or system help reply.");
            replay_codes.Add("212","Directory status.");
            replay_codes.Add("213","File status.");
            replay_codes.Add("214","Help message.");
            replay_codes.Add("215","NAME system type.");
            replay_codes.Add("220","Service ready for new user.");
            replay_codes.Add("221","Service closing control connection.");
            replay_codes.Add("225","Data connection open; no transfer in progress.");
            replay_codes.Add("226","Closing data connection.");
            replay_codes.Add("227","Entering Passive Mode");
            replay_codes.Add("230","User logged in, proceed.");
            replay_codes.Add("250","Requested file action okay, completed.");
            replay_codes.Add("257","PATHNAME created.");
            replay_codes.Add("331","User name okay, need password.");
            replay_codes.Add("332","Need account for login.");
            replay_codes.Add("350","Requested file action pending further information.");
            replay_codes.Add("421","Service not available, closing control connection.");
            replay_codes.Add("425","Can't open data connection.");
            replay_codes.Add("426","Connection closed; transfer aborted.");
            replay_codes.Add("450","Requested file action not taken.");
            replay_codes.Add("451","File unavailable");
            replay_codes.Add("452","Requested action not taken.");
            replay_codes.Add("500","Syntax error, command unrecognized.");
            replay_codes.Add("501","Syntax error in parameters or arguments.");
            replay_codes.Add("502","Command not implemented.");
            replay_codes.Add("503","Bad sequence of commands.");
            replay_codes.Add("504","Command not implemented for that parameter.");
            replay_codes.Add("530","Not logged in.");
            replay_codes.Add("532","Need account for storing files.");
            replay_codes.Add("550","Requested action not taken.");
            replay_codes.Add("551","Requested action aborted: page type unknown.");
            replay_codes.Add("552","Requested file action aborted.");
            replay_codes.Add("553","File name not allowed.");
        }


        static private string ParseLine(string input)
        {                                    
            string[] field = input.Split(',');
            string direction = field[0] + " ";
            string data = field[1];
            string line = "";

            /* Client -> Server */
            if (direction.StartsWith(">>")) 
            {
                foreach (string s in commands.Keys)
                {
                    if (data.StartsWith(s))
                    {
                        line += String.Format("{0} {1} ({2})", direction, s, commands[s]);
                    }                 
                }
            }
                
            /* Server -> Client */
            if (direction.StartsWith("<<"))
            {
                foreach (string s in replay_codes.Keys)
                {
                    if (data.StartsWith(s))
                    {
                        line += String.Format("{0} {1} ({2})", direction, s, replay_codes[s]);
                    }                 
                }
            }

            return line;
        }


        static void Usage()
        {
            Console.WriteLine("Parse csv file (with TCP stream) and return txt file with FTP relation");
            Console.WriteLine("Usage: FTPParser \"input_file.csv\"");
            return;
        }


        static void Main(string[] args)
        {
            if (args.Length != 1)
            {
                Usage();
                return;
            }

            // open files
            if (!File.Exists(args[0]))
            {
                Console.WriteLine("File not found!");
                Usage();
                return;
            }

            init();

            csvr = new CSVReader(args[0]);
            sw = new StreamWriter("FTP_data.txt");

            Console.Write("Processing");

            while (!csvr.EndOfStream()) // for each line
            {
                string pl = ParseLine(csvr.ReadLine());
                if (pl.Trim() != "")
                {
                    sw.WriteLine(pl);    
                }
                
                Console.Write(".");
            }

            csvr.Close();         
            sw.Close();

            Console.WriteLine("\nParse complete, press any key to exit...");
            Console.ReadKey();
        }
    }    
}
